/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
  i-net software 1998-2013

*/
using System.Drawing.Printing;
namespace Inet.Viewer.Data
{
    /// <summary>
    /// Orientation of the page
    /// </summary>
    public enum PageOrientation
    {
        /// <summary>
        /// Defaul paper orientation
        /// </summary>
        DefaultPaperOrientation = 0,

        /// <summary>
        /// Potrait orientation
        /// </summary>
        Portrait = 1,

        /// <summary>
        /// Landscape orientation
        /// </summary>
        Landscape = 2
    }


    /// <summary>
    /// Stores all status information for one page. This information is transmitted for each page.    
    /// </summary>
    public class PageInfo
    {
        /// <summary>
        /// 1 inch = 1440 twips, 1/100 inch = TWIPS_OF_INCH100 twips 
        /// </summary>
        private const float TwipsOfInch100 = 14.4F;

        private int pageNr;
        private int pageWidth;
        private int pageHeight;
        private int westBorder;
        private int northBorder;
        private int eastBorder;
        private int southBorder;
        private int pageAlign;
        private long timestamp;

        /// <summary>
        /// Gets and sets the number of this page.
        /// </summary>
        internal virtual int PageNr
        {
            get
            {
                return pageNr;
            }
            set
            {
                this.pageNr = value;
            }
        }

        /// <summary>        
        /// Gets and sets the visible Page with without the border in twips.
        /// </summary>
        internal virtual int PageWidth
        {
            get
            {
                return pageWidth;
            }
            set
            {
                this.pageWidth = value;
            }
        }

        /// <summary>
        /// Gets and sets the visisble page without the border in Twips
        /// </summary>
        internal virtual int PageHeight
        {
            get
            {
                return pageHeight;
            }
            set
            {
                this.pageHeight = value;
            }
        }

        /// <summary>        
        /// Gets and sets the left border in twips.
        /// </summary>
        internal virtual int BorderLeft
        {
            get
            {
                return westBorder;
            }
            set
            {
                this.westBorder = value;
            }
        }

        /// <summary>
        /// Gets and sets the top border in twips.
        /// </summary>
        internal virtual int BorderTop
        {
            get
            {
                return northBorder;
            }
            set
            {
                this.northBorder = value;
            }
        }

        /// <summary>    
        /// Gets and sets the right border in twips.
        /// </summary>
        internal virtual int BorderRight
        {
            get
            {
                return eastBorder;
            }
            set
            {
                this.eastBorder = value;
            }
        }

        /// <summary>   
        /// Gets and sets the bottom border in twips.
        /// </summary>
        internal virtual int BorderBottom
        {
            get
            {
                return southBorder;
            }
            set
            {
                this.southBorder = value;
            }
        }

        /// <summary>
        /// <see cref="PageOrientation"/>
        /// </summary>
        internal virtual int PageAlign
        {
            get
            {
                return pageAlign;
            }
            set
            {
                this.pageAlign = value;
            }
        }

        /// <summary>
        /// Gets and sets the timestamp.
        /// </summary>
        internal virtual long Timestamp
        {
            get
            {
                return timestamp;
            }
            set
            {
                this.timestamp = value;
            }
        }

        /// <summary>
        /// Gets the visible PageWidth with Borders
        /// <see cref="PageWidth"/> + <see cref="BorderLeft"/><see cref="BorderRight"/> in TWIPS
        /// </summary>
        internal int Width
        {
            get
            {
                return this.PageWidth + this.BorderLeft + this.BorderRight;
            }
        }

        /// <summary>
        /// Gets the visible Page Height with Borders
        /// <see cref="PageHeight"/> + <see cref="BorderTop"/><see cref="BorderBottom"/> in TWIPS
        /// </summary>
        internal int Height
        {
            get
            {
                return this.PageHeight + this.BorderTop + this.BorderBottom;
            }
        }

        /// <summary>
        /// Gets the margins for printing.
        /// </summary>
        public Margins PrintMargins
        {
            get
            {
                return new Margins((int)(BorderLeft / TwipsOfInch100), (int)(BorderRight / TwipsOfInch100), (int)(BorderTop / TwipsOfInch100), (int)(BorderBottom / TwipsOfInch100));
            }
        }
    }
}